<?php

namespace App\Http\Controllers\Merchant;

use App\Constants\NotificationConst;
use App\Constants\PaymentGatewayConst;
use App\Http\Controllers\Controller;
use App\Http\Helpers\NotificationHelper;
use App\Http\Helpers\PushNotificationHelper;
use Illuminate\Http\Request;
use App\Models\Admin\Currency;
use App\Models\Admin\PaymentGateway;
use App\Models\Admin\PaymentGatewayCurrency;
use App\Models\Transaction;
use Illuminate\Support\Facades\Validator;
use App\Traits\ControlDynamicInputFields;
use Exception;
use Illuminate\Support\Facades\DB;
use Jenssegers\Agent\Agent;
use App\Models\Admin\BasicSettings;
use App\Models\Merchants\MerchantNotification;
use App\Models\Merchants\MerchantWallet;
use App\Notifications\Admin\ActivityNotification;
use App\Notifications\User\Withdraw\WithdrawMail;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Auth;

class MoneyOutController extends Controller
{
    use ControlDynamicInputFields;
    public function index()
    {
        $page_title = __("Withdraw Money");
        $user_wallets = MerchantWallet::auth()->get();
        $wallet_currencies = Currency::whereIn('id',$user_wallets->pluck('currency_id')->toArray())->get();
        $payment_gateways = PaymentGatewayCurrency::whereHas('gateway', function ($gateway) {
            $gateway->where('slug', PaymentGatewayConst::money_out_slug());
            $gateway->where('status', 1);
        })->get();

        $transactions = Transaction::merchantAuth()->moneyOut()->orderByDesc("id")->latest()->take(10)->get();

        return view('merchant.sections.withdraw.index',compact('page_title','payment_gateways','transactions','wallet_currencies'));
    }

   public function paymentInsert(Request $request){
        $validated = Validator::make($request->all(),[
            'amount'            => 'required|numeric|gt:0',
            'gateway'           => 'required',
            'currency'     => "required|string|exists:currencies,code",
        ])->validate();

        $user = userGuard()['user'];
        $amount = $validated['amount'];

        $userWallet = MerchantWallet::where('merchant_id',$user->id)->whereHas("currency",function($q) use ($validated) {
            $q->where("code",$validated['currency'])->active();
        })->active()->first();
        if(!$userWallet) return back()->with(['error' => [__("Merchant wallet not found!")]]);

        $gate =PaymentGatewayCurrency::whereHas('gateway', function ($gateway) {
            $gateway->where('slug', PaymentGatewayConst::money_out_slug());
            $gateway->where('status', 1);
        })->where('alias',$request->gateway)->first();
        if (!$gate) {
            return back()->with(['error' => [__("Gateway is not available right now! Please contact with system administration")]]);
        }

        $baseCurrency = Currency::default();
        if (!$baseCurrency) {
            return back()->with(['error' => [__("Default currency not found")]]);
        }

        $charges = $this->withdrawCharges($validated['amount'],$userWallet,$gate);
        $min_amount = $gate->min_limit / $charges['exchange_rate'];
        $max_amount = $gate->max_limit / $charges['exchange_rate'];

        if((float)$charges['requested_amount'] < getAmount($min_amount,4) || (float)$charges['requested_amount'] > getAmount($max_amount,4)) {
            return back()->with(['error' => [__("Please follow the transaction limit")]]);
        }

        if($charges['payable'] > $userWallet->balance) {
            return back()->with(['error' => [__("Sorry, insufficient balance")]]);
        }

        $data['merchant_id']= $user->id;
        $data['gateway_name']= $gate->gateway->name;
        $data['gateway_type']= $gate->gateway->type;
        $data['wallet_id']= $userWallet->id;
        $data['trx_id']= 'MO'.getTrxNum();
        $data['amount'] =  $amount;
        $data['gateway_id'] = $gate->gateway->id;
        $data['gateway_currency_id'] = $gate->id;
        $data['gateway_currency'] = strtoupper($gate->currency_code);
        $data['charges'] = $charges;

        session()->put('moneyoutData', $data);
        return redirect()->route('merchant.withdraw.preview');
   }
   public function preview(){
    $moneyOutData = (object)session()->get('moneyoutData');
    $moneyOutDataExist = session()->get('moneyoutData');
    if($moneyOutDataExist  == null){
        return redirect()->route('merchant.withdraw.index');
    }
    $gateway = PaymentGateway::where('id', $moneyOutData->gateway_id)->first();
    if($gateway->type == "AUTOMATIC"){
        $page_title = "Withdraw Via ".$gateway->name;
        if(strtolower($gateway->name) == "flutterwave"){
            $credentials = $gateway->credentials;
            $data = null;
            foreach ($credentials as $object) {
                $object = (object)$object;
                if ($object->label === "Secret key") {
                    $data = $object;
                    break;
                }
            }
            $countries = get_all_countries();
            $currency =  $moneyOutData->gateway_currency;
            $country = Collection::make($countries)->first(function ($item) use ($currency) {
                return $item->currency_code === $currency;
            });

            $allBanks = getFlutterwaveBanks($country->iso2);
            return view('merchant.sections.withdraw.automatic.'.strtolower($gateway->name),compact('page_title','gateway','moneyOutData','allBanks'));
        }else{
            return back()->with(['error' => [__("Something went wrong! Please try again.")]]);
        }
    }else{
        $page_title = __("Withdraw Via")." ".$gateway->name;
        return view('merchant.sections.withdraw.preview',compact('page_title','gateway','moneyOutData'));
    }

   }
   public function confirmMoneyOut(Request $request){
    $basic_setting = BasicSettings::first();
    $moneyOutData = (object)session()->get('moneyoutData');
    $gateway = PaymentGateway::where('id', $moneyOutData->gateway_id)->first();
    $payment_fields = $gateway->input_fields ?? [];

    $validation_rules = $this->generateValidationRules($payment_fields);
    $payment_field_validate = Validator::make($request->all(),$validation_rules)->validate();
    $get_values = $this->placeValueWithFields($payment_fields,$payment_field_validate);
        try{
            $get_values =[
                'user_data' => $get_values,
                'charges' => $moneyOutData->charges,
            ];
            //send notifications
            $user = auth()->user();
            $inserted_id = $this->insertRecordManual($moneyOutData,$gateway,$get_values,$reference= null,PaymentGatewayConst::STATUSPENDING);
            $this->insertChargesManual($moneyOutData,$inserted_id);
            $this->adminNotification($moneyOutData,PaymentGatewayConst::STATUSPENDING);
            $this->insertDeviceManual($moneyOutData,$inserted_id);
            session()->forget('moneyoutData');
            try{
                if( $basic_setting->merchant_email_notification == true){
                   $user->notify(new WithdrawMail($user,$moneyOutData));
               }
            }catch(Exception $e){

            }
            return redirect()->route("merchant.withdraw.index")->with(['success' => [__('Withdraw money request send to admin successful')]]);
        }catch(Exception $e) {
            return redirect()->route("merchant.withdraw.index")->with(['error' => [__("Something went wrong! Please try again.")]]);
        }

   }
   public function confirmMoneyOutAutomatic(Request $request){
    $basic_setting = BasicSettings::first();
    if($request->gateway_name == 'flutterwave'){
        $request->validate([
            'bank_name' => 'required',
            'account_number' => 'required'
        ]);
        $moneyOutData = (object)session()->get('moneyoutData');
        $gateway = PaymentGateway::where('id', $moneyOutData->gateway_id)->first();

        $credentials = $gateway->credentials;
        $secret_key = getPaymentCredentials($credentials,'Secret key');
        $base_url = getPaymentCredentials($credentials,'Base Url');
        $callback_url = url('/').'/flutterwave/withdraw_webhooks';

        $ch = curl_init();
        $url =  $base_url.'/transfers';
        $reference =  generateTransactionReference();
        $data = [
            "account_bank" => $request->bank_name,
            "account_number" => $request->account_number,
            "amount" => $moneyOutData->charges['will_get'],
            "narration" => "Withdraw from wallet",
            "currency" =>$moneyOutData->gateway_currency,
            "reference" => $reference,
            "callback_url" => $callback_url,
            "debit_currency" => $moneyOutData->gateway_currency
        ];
        $headers = [
            'Authorization: Bearer '.$secret_key,
            'Content-Type: application/json'
        ];

        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

        $response = curl_exec($ch);
        if (curl_errno($ch)) {
            return back()->with(['error' => [curl_error($ch)]]);
        } else {
            $result = json_decode($response,true);
            if($result['status'] && $result['status'] == 'success'){
                try{
                    $get_values =[
                        'user_data' => null,
                        'charges' => $moneyOutData->charges,
                    ];
                    $user = auth()->user();
                    $inserted_id = $this->insertRecordManual($moneyOutData,$gateway,$get_values,$reference,PaymentGatewayConst::STATUSWAITING);
                    $this->insertChargesAutomatic($moneyOutData,$inserted_id);
                    $this->adminNotification($moneyOutData,PaymentGatewayConst::STATUSSUCCESS);
                    $this->insertDeviceManual($moneyOutData,$inserted_id);
                    session()->forget('moneyoutData');
                    try{
                        if( $basic_setting->merchant_email_notification == true){
                            $user->notify(new WithdrawMail($user,$moneyOutData));
                        }
                    }catch(Exception $e){

                    }
                    return redirect()->route("merchant.withdraw.index")->with(['success' => [__('Withdraw money request send successful')]]);
                }catch(Exception $e) {
                    return back()->with(['error' => [__("Something went wrong! Please try again.")]]);
                }

            }else{
                return back()->with(['error' => [$result['message']]]);
            }
        }

        curl_close($ch);

    }else{
        return back()->with(['error' => [__("Invalid request,please try again later")]]);
    }
   }

   //check flutterwave banks
   public function checkBanks(Request $request){
    $bank_account = $request->account_number;
    $bank_code = $request->bank_code;
    $exist['data'] = (checkBankAccount($secret_key = null,$bank_account,$bank_code));
    return response( $exist);
   }

    public function insertRecordManual($moneyOutData,$gateway,$get_values,$reference,$status) {
        $trx_id = $moneyOutData->trx_id ??'MO'.getTrxNum();
        $authWallet = MerchantWallet::where('id',$moneyOutData->wallet_id)->where('merchant_id',$moneyOutData->merchant_id)->first();
        $afterCharge = ($authWallet->balance - ($moneyOutData->charges['payable']??$moneyOutData->amount));

        DB::beginTransaction();
        try{
            $id = DB::table("transactions")->insertGetId([
                'merchant_id'                   => $moneyOutData->merchant_id,
                'merchant_wallet_id'            => $moneyOutData->wallet_id,
                'payment_gateway_currency_id'   => $moneyOutData->gateway_currency_id,
                'type'                          => PaymentGatewayConst::TYPEMONEYOUT,
                'trx_id'                        => $trx_id,
                'request_amount'                => $moneyOutData->amount,
                'payable'                       => $moneyOutData->charges['will_get'],
                'available_balance'             => $afterCharge,
                'remark'                        => ucwords(remove_speacial_char(PaymentGatewayConst::TYPEMONEYOUT," ")) . " by " .$gateway->name,
                'details'                       => json_encode($get_values),
                'status'                        => $status,
                'callback_ref'                  => $reference??null,
                'created_at'                    => now(),
            ]);

            $this->updateWalletBalanceManual($authWallet,$afterCharge);


            DB::commit();
        }catch(Exception $e) {
            DB::rollBack();
            throw new Exception(__("Something went wrong! Please try again."));
        }
        return $id;
    }


    public function updateWalletBalanceManual($authWalle,$afterCharge) {
        $authWalle->update([
            'balance'   => $afterCharge,
        ]);
    }
    public function insertChargesAutomatic($moneyOutData,$id) {
        if(Auth::guard(get_auth_guard())->check()){
            $merchant = auth()->guard(get_auth_guard())->user();
        }
        DB::beginTransaction();
        try{
            DB::table('transaction_charges')->insert([
                'transaction_id'    => $id,
                'percent_charge'    => $moneyOutData->charges['percent_charge'],
                'fixed_charge'      => $moneyOutData->charges['fixed_charge'],
                'total_charge'      => $moneyOutData->charges['total_charge'],
                'created_at'        => now(),
            ]);
            DB::commit();

            //notification
            $notification_content = [
                'title'         => __("Withdraw Money"),
                'message'       => __("Your Withdraw Request")." " .get_amount($moneyOutData->amount,$moneyOutData->charges['wallet_cur_code'],4)." ".__("Successful"),
                'image'         => get_image($merchant->image,'merchant-profile'),
            ];

            MerchantNotification::create([
                'type'      => NotificationConst::MONEY_OUT,
                'merchant_id'  =>  $moneyOutData->merchant_id,
                'message'   => $notification_content,
            ]);

            //Push Notifications
            try{
                (new PushNotificationHelper())->prepare([$merchant->id],[
                    'title' => $notification_content['title'],
                    'desc'  => $notification_content['message'],
                    'user_type' => 'merchant',
                ])->send();
            }catch(Exception $e) {}
        }catch(Exception $e) {
            DB::rollBack();
            throw new Exception(__("Something went wrong! Please try again."));
        }
    }
    public function insertChargesManual($moneyOutData,$id) {
        if(Auth::guard(get_auth_guard())->check()){
            $merchant = auth()->guard(get_auth_guard())->user();
        }
        DB::beginTransaction();
        try{
            DB::table('transaction_charges')->insert([
                'transaction_id'    => $id,
                'percent_charge'    => $moneyOutData->charges['percent_charge'],
                'fixed_charge'      => $moneyOutData->charges['fixed_charge'],
                'total_charge'      => $moneyOutData->charges['total_charge'],
                'created_at'        => now(),
            ]);
            DB::commit();

            //notification
            $notification_content = [
                'title'         => __("Withdraw Money"),
                'message'       => __("Your Withdraw Request Send To Admin")." " .get_amount($moneyOutData->amount,$moneyOutData->charges['wallet_cur_code'],4)." ".__("Successful"),
                'image'         => get_image($merchant->image,'merchant-profile'),
            ];

            MerchantNotification::create([
                'type'      => NotificationConst::MONEY_OUT,
                'merchant_id'  =>  $moneyOutData->merchant_id,
                'message'   => $notification_content,
            ]);

            //Push Notifications
            try{
                (new PushNotificationHelper())->prepare([$merchant->id],[
                    'title' => $notification_content['title'],
                    'desc'  => $notification_content['message'],
                    'user_type' => 'merchant',
                ])->send();
            }catch(Exception $e) {}
        }catch(Exception $e) {
            DB::rollBack();
            throw new Exception(__("Something went wrong! Please try again."));
        }
    }

    public function insertDeviceManual($output,$id) {
        $client_ip = request()->ip() ?? false;
        $location = geoip()->getLocation($client_ip);
        $agent = new Agent();

        // $mac = exec('getmac');
        // $mac = explode(" ",$mac);
        // $mac = array_shift($mac);
        $mac = "";

        DB::beginTransaction();
        try{
            DB::table("transaction_devices")->insert([
                'transaction_id'=> $id,
                'ip'            => $client_ip,
                'mac'           => $mac,
                'city'          => $location['city'] ?? "",
                'country'       => $location['country'] ?? "",
                'longitude'     => $location['lon'] ?? "",
                'latitude'      => $location['lat'] ?? "",
                'timezone'      => $location['timezone'] ?? "",
                'browser'       => $agent->browser() ?? "",
                'os'            => $agent->platform() ?? "",
            ]);
            DB::commit();
        }catch(Exception $e) {
            DB::rollBack();
            throw new Exception(__("Something went wrong! Please try again."));
        }
    }

    //admin notification global(Agent & User)
    public function adminNotification($data,$status){
        $user = auth()->guard(userGuard()['guard'])->user();
        $exchange_rate = " 1 ". $data->charges['wallet_cur_code'].' = '. get_amount($data->charges['exchange_rate'],$data->charges['gateway_cur_code'],4);
        if($status == PaymentGatewayConst::STATUSSUCCESS){
            $status ="success";
        }elseif($status == PaymentGatewayConst::STATUSPENDING){
            $status ="Pending";
        }elseif($status == PaymentGatewayConst::STATUSHOLD){
            $status ="Hold";
        }elseif($status == PaymentGatewayConst::STATUSWAITING){
            $status ="Waiting";
        }elseif($status == PaymentGatewayConst::STATUSPROCESSING){
            $status ="Processing";
        }elseif($status == PaymentGatewayConst::STATUSFAILD){
            $status ="Failed";
        }

        $notification_content = [
            //email notification
            'subject' =>__("Withdraw Money")." (".userGuard()['type'].")",
            'greeting' =>__("Withdraw Money Via")." ".$data->gateway_name.' ('.$data->gateway_type.' )',
            'email_content' =>__("web_trx_id")." : ".$data->trx_id."<br>".__("request Amount")." : ".get_amount($data->amount,$data->charges['wallet_cur_code']??get_default_currency_code())."<br>".__("Exchange Rate")." : ". $exchange_rate."<br>".__("Fees & Charges")." : ". get_amount($data->charges['total_charge']??$data->gateway_charge,$data->charges['wallet_cur_code']??$data->gateway_currency)."<br>".__("Total Payable Amount")." : ".get_amount($data->charges['payable']??$data->payable,$data->charges['wallet_cur_code']??get_default_currency_code())."<br>".__("Will Get")." : ".get_amount($data->charges['will_get']??$data->will_get,$data->charges['gateway_cur_code']??$data->gateway_currency,2)."<br>".__("Status")." : ".__($status),
            //push notification
            'push_title' =>  __("Withdraw Money")." (".userGuard()['type'].")",
            'push_content' => __('web_trx_id')." ".$data->trx_id." ". __("Withdraw Money").' '.get_amount($data->amount,$data->charges['wallet_cur_code']).' '.__('By').' '.$data->gateway_name.' ('.$user->username.')',

            //admin db notification
            'notification_type' =>  NotificationConst::MONEY_OUT,
            'trx_id' => $data->trx_id,
            'admin_db_title' =>  "Withdraw Money"." (".userGuard()['type'].")",
            'admin_db_message' =>  "Withdraw Money".' '.get_amount($data->amount,$data->charges['wallet_cur_code']).' '.'By'.' '.$data->gateway_name.' ('.$user->username.')'
        ];

        try{
            //notification
            (new NotificationHelper())->admin(['admin.money.out.index','admin.money.out.pending','admin.money.out.complete','admin.money.out.canceled','admin.money.out.details','admin.money.out.approved','admin.money.out.rejected','admin.money.out.export.data'])
                                    ->mail(ActivityNotification::class, [
                                        'subject'   => $notification_content['subject'],
                                        'greeting'  => $notification_content['greeting'],
                                        'content'   => $notification_content['email_content'],
                                    ])
                                    ->push([
                                        'user_type' => "admin",
                                        'title' => $notification_content['push_title'],
                                        'desc'  => $notification_content['push_content'],
                                    ])
                                    ->adminDbContent([
                                        'type' => $notification_content['notification_type'],
                                        'title' => $notification_content['admin_db_title'],
                                        'message'  => $notification_content['admin_db_message'],
                                    ])
                                    ->send();


        }catch(Exception $e) {}

    }
    public function withdrawCharges($sender_amount,$userWallet,$gate) {

        $exchange_rate = $gate->rate / $userWallet->currency->rate;

        $data['exchange_rate']          = $exchange_rate;
        $data['requested_amount']         = $sender_amount;
        $data['gateway_cur_code']         =  $gate->currency_code;
        $data['gateway_cur_rate']         =  $gate->rate;
        $data['wallet_cur_code']         =  $userWallet->currency->code;
        $data['wallet_cur_rate']         =  $userWallet->currency->rate;
        $data['will_get']               = $sender_amount * $exchange_rate;
        $data['conversion_amount']      = $sender_amount * $exchange_rate;
        $data['percent_charge']         = ($sender_amount / 100) * $gate->percent_charge ?? 0;
        $data['fixed_charge']           =  $gate->fixed_charge/$exchange_rate ?? 0;
        $data['total_charge']           = $data['percent_charge'] + $data['fixed_charge'];
        $data['sender_wallet_balance']  = $userWallet->balance;
        $data['payable']                = $sender_amount + $data['total_charge'];
        $data['default_currency']       = get_default_currency_code();

        return $data;
    }
}
